#include "variable.h"

#include "slave.h"

template <class T>
static QByteArray toByteArray(T value) {
	return QByteArray((const char*) &value, sizeof(value));
}

/***** EtherCATInput *****/

EtherCATInput::EtherCATInput (EtherCATSlave* slave, int byteIndex, QByteArray value, QObject* parent)
	: QObject(parent), slave(slave), byteIndex(byteIndex), value(value)
{
	connect(this, &EtherCATInput::threadSafeUpdate, this, &EtherCATInput::update);
	slave->threadSafeRegisterVar(byteIndex, this, value);
}

void EtherCATInput::update (QByteArray value) {
	this->value = value;
	reportChange();
}


/***** EtherCATInputUInt8 *****/

EtherCATInputUInt8::EtherCATInputUInt8 (EtherCATSlave* slave, int byteIndex, quint8 initialValue, QObject* parent)
		: EtherCATInput(slave, byteIndex, toByteArray(initialValue), parent)
{}

quint8 EtherCATInputUInt8::getValue () {
	return *((quint8*) value.data());
}

void EtherCATInputUInt8::reportChange () {
	emit changed(getValue());
}


/***** EtherCATInputUInt16 *****/

EtherCATInputUInt16::EtherCATInputUInt16 (EtherCATSlave* slave, int byteIndex, quint16 initialValue, QObject* parent)
		: EtherCATInput(slave, byteIndex, toByteArray(initialValue), parent)
{}

quint16 EtherCATInputUInt16::getValue () {
	return *((quint16*) value.data());
}

void EtherCATInputUInt16::reportChange () {
	emit changed(getValue());
}


/***** EtherCATInputUInt64 *****/

EtherCATInputUInt64::EtherCATInputUInt64 (EtherCATSlave* slave, int byteIndex, quint64 initialValue, QObject* parent)
		: EtherCATInput(slave, byteIndex, toByteArray(initialValue), parent)
{}

quint64 EtherCATInputUInt64::getValue () {
	return *((quint64*) value.data());
}

void EtherCATInputUInt64::reportChange () {
	emit changed(getValue());
}


/***** EtherCATInputFloat *****/

EtherCATInputFloat::EtherCATInputFloat (EtherCATSlave* slave, int byteIndex, float initialValue, QObject* parent)
		: EtherCATInput(slave, byteIndex, toByteArray(initialValue), parent)
{}

float EtherCATInputFloat::getValue () {
	return *((float*) value.data());
}

void EtherCATInputFloat::reportChange () {
	emit changed(getValue());
}


/***** EtherCATOutput *****/

EtherCATOutput::EtherCATOutput (EtherCATSlave* slave, int byteIndex, QObject* parent) :
	QObject(parent), slave(slave), byteIndex(byteIndex)
{}


/***** EtherCATOutputUInt8 *****/

EtherCATOutputUInt8::EtherCATOutputUInt8 (EtherCATSlave* slave, int byteIndex, quint8 initialValue, QObject* parent)
	: EtherCATOutput(slave, byteIndex, parent)
{
	setValue(initialValue);
}

void EtherCATOutputUInt8::setValue (quint8 value) {
	this->value = value;
	slave->threadSafeSetValue(byteIndex, toByteArray(value));
}

void EtherCATOutputUInt8::setBit (int index, bool enabled) {
	quint8 mask = 1 << index;
	setValue(enabled ? value | mask : value & ~mask);
}

/***** EtherCATOutputUInt16 *****/

EtherCATOutputUInt16::EtherCATOutputUInt16 (EtherCATSlave* slave, int byteIndex, quint16 initialValue, QObject* parent)
	: EtherCATOutput(slave, byteIndex, parent)
{
	setValue(initialValue);
}

void EtherCATOutputUInt16::setValue (quint16 value) {
	this->value = value;
	slave->threadSafeSetValue(byteIndex, toByteArray(value));
}

void EtherCATOutputUInt16::setBit (int index, bool enabled) {
	quint16 mask = 1 << index;
	setValue(enabled ? value | mask : value & ~mask);
}


/***** EtherCATOutputUInt64 *****/

EtherCATOutputUInt64::EtherCATOutputUInt64 (EtherCATSlave* slave, int byteIndex, quint64 initialValue, QObject* parent)
	: EtherCATOutput(slave, byteIndex, parent)
{
	setValue(initialValue);
}

void EtherCATOutputUInt64::setValue (quint64 value) {
	this->value = value;
	slave->threadSafeSetValue(byteIndex, toByteArray(value));
}


/***** EtherCATOutputFloat *****/

EtherCATOutputFloat::EtherCATOutputFloat (EtherCATSlave* slave, int byteIndex, float initialValue, QObject* parent)
	: EtherCATOutput(slave, byteIndex, parent)
{
	setValue(initialValue);
}

void EtherCATOutputFloat::setValue (float value) {
	this->value = value;
	slave->threadSafeSetValue(byteIndex, toByteArray(value));
}
